"""VR Demo Recorder Window."""
from pathlib import Path
from typing import Optional

from bigym.action_modes import JointPositionActionMode, PelvisDof, DEFAULT_DOFS
from tools.shared.base_window import BaseWindow
import dearpygui.dearpygui as dpg

from tools.shared.utils import ENVIRONMENTS
from vr.viewer.control_profiles.standard_control_profile import StandardControlProfile
from vr.viewer.vr_viewer import VRViewer


class DemoRecorderWindow(BaseWindow):
    """VR Demo Recorder Window."""

    def __init__(self):
        """Init."""
        self._target_dir: Path = Path(__file__).parent / "demo"
        self._viewer: Optional[VRViewer] = None
        super().__init__()

    def _setup_ui(self):
        dpg.add_text("Environment:")
        self._env_listbox = dpg.add_listbox(
            num_items=20, items=list(sorted(ENVIRONMENTS.keys())), width=-1
        )
        dpg.add_spacer()

        dpg.add_text("Pelvis DOFs:")
        self._dof_checkboxes: dict[PelvisDof, int] = {}
        with dpg.group(horizontal=True):
            for dof in PelvisDof:
                is_default = dof in DEFAULT_DOFS
                self._dof_checkboxes[dof] = dpg.add_checkbox(
                    label=dof.value, default_value=is_default
                )
                dpg.add_spacer()

        dpg.add_text("Output Directory:")
        with dpg.group(horizontal=True):
            dpg.add_button(label="Change", callback=self._select_directory_callback)
            self._out_directory_label = dpg.add_text(str(self._target_dir))
        dpg.add_spacer()

        self._record_button = dpg.add_button(
            label="Record", callback=self._record, width=-1
        )
        self._stop_button = dpg.add_button(
            label="Stop", show=False, width=-1, callback=self._stop
        )

    def _select_directory_callback(self):
        self._select_directory(self._target_dir, self._on_target_dir_selected)

    def _on_target_dir_selected(self, target_path: Path):
        self._target_dir = target_path
        dpg.set_value(self._out_directory_label, str(self._target_dir))

    def _record(self):
        if self._viewer:
            return

        dpg.hide_item(self._record_button)
        dpg.show_item(self._stop_button)

        self._start_viewer()

    def _stop(self):
        if not self._viewer:
            return

        dpg.hide_item(self._stop_button)
        dpg.show_item(self._record_button)

        self._viewer = None

    def _start_viewer(self):
        floating_dofs = [
            dof
            for dof, checkbox in self._dof_checkboxes.items()
            if dpg.get_value(checkbox)
        ]
        self._viewer = VRViewer(
            env_cls=ENVIRONMENTS[dpg.get_value(self._env_listbox)],
            action_mode=JointPositionActionMode(
                absolute=True, floating_base=True, floating_dofs=floating_dofs
            ),
            control_profile=StandardControlProfile(),
            demo_directory=self._target_dir,
        )
        self._viewer.run()
